class Compiler:
    def __init__(self, program: dict, mode: str) -> None:
        self.program = program
        self.mode = mode
        self.operation_view_dsl = []
        self.product_view_dsl = []
    
    def compile(self):
        if self.mode == "dsl":
            self.operation_view_dsl = self.program
            return self.check_dsl_program()
        elif self.mode == "multi-dsl":
            self.operation_view_dsl, self.product_view_dsl = self.__split_dsl(self.program)
            return self.check_multi_dsl_program()
    
    def check_dsl_program(self):
        feedback = []
        output_stack = set()
        input_stack = set()
        for i, step in enumerate(self.operation_view_dsl):
            operation = step.get("Operation")
            precond = step.get("Precond", {})
            postcond = step.get("Postcond", {})
            input_products = precond.get("SlotArg", [])
            output_products = postcond.get("EmitArg", [])
            if i == 0:
                output_stack.update(output_products)
                continue
            input_stack.update(input_products)
            for product in input_products:
                if product not in output_stack:
                    error_info = f"Error: The product '{product}' required by operation '{operation}' at step {i} is not available from previous steps."
                    feedback.append(error_info)
            output_stack.update(output_products)

        for i, step in enumerate(self.operation_view_dsl[:-1]):
            operation = step.get("Operation")
            postcond = step.get("Postcond", {})
            output_products = postcond.get("EmitArg", [])
            for product in output_products:
                if product not in input_stack:
                    error_info = f"Error: The product '{product}' generated by operation '{operation}' at step {i} is not used in subsequent steps."
                    feedback.append(error_info)

        if len(feedback) == 0:
            feedback.append("DSL program check passed.")
        
        return feedback

    def check_multi_dsl_program(self):
        feedback = []
        operation_map = {}
        for index, op_step in enumerate(self.operation_view_dsl):
            operation = op_step.get("Operation")
            precond_products = op_step.get("Precond", {}).get("SlotArg", [])
            postcond_products = op_step.get("Postcond", {}).get("EmitArg", [])
            operation_key = f"{operation}_{index}"
            operation_map[operation_key] = {
                "Precond": precond_products,
                "Postcond": postcond_products
            }
        # print(operation_map)
        for product_step in self.product_view_dsl:
            product_ref = product_step.get("FlowUnit", {}).get("RefName")
            product_com = product_step.get("FlowUnit", {}).get("Component")
            pred_operation = product_step.get("Pred")
            succ_operation = product_step.get("Succ")
            # 是否是前置动作的输出
            if pred_operation:
                for operation_key, operation_details in operation_map.items():
                    operation_name = operation_key.split("_")[0]
                    # print(operation_name, pred_operation, product_ref, product_com)
                    if operation_name == pred_operation and (product_ref in operation_details["Postcond"] or product_com in operation_details["Postcond"]):
                        break
                else:
                    error_info = f"Error: Product '{product_ref}' was not correctly generated by operation '{pred_operation}'."
                    feedback.append(error_info)
            # 是否是后置动作的输入
            if succ_operation:
                for operation_key, operation_details in operation_map.items():
                    operation_name = operation_key.split("_")[0]
                    if operation_name == succ_operation and (product_ref in operation_details["Precond"] or product_com in operation_details["Precond"]):
                        break
                else:
                    error_info = f"Error: Product '{product_ref}' was not correctly consumed by operation '{succ_operation}'."
                    feedback.append(error_info)
        
        if len(feedback) == 0:
            feedback.append("DSL program cross-verification passed.")
        
        return feedback

    def __split_dsl(self, program):
        return (
            [step_dict for step_dict in program if "Execution" in step_dict],
            [step_dict for step_dict in program if "FlowUnit" in step_dict]
        )